#ifndef photochemistry_node_helpers_hpp
#define photochemistry_node_helpers_hpp

#include <napi.h>
#include <stdio.h>
#include <iostream>
#include "photochemistry/image.hpp"
#include "photochemistry/process.hpp"
#include "photochemistry/utf8.hpp"
#include <opencv2/opencv.hpp>
#include <future>
#include <iostream>
#include <memory>


inline Napi::Buffer<uint8_t> U8ArrayView(Napi::Env env, const uint8_t* data, size_t elements) {
    return Napi::Buffer<uint8_t>::Copy(env, data, elements);
}

inline Napi::Buffer<uint16_t> U16ArrayView(Napi::Env env, const uint16_t* data, size_t elements) {
    return Napi::Buffer<uint16_t>::Copy(env, data, elements * sizeof(uint16_t));
}

inline Napi::Buffer<float> FloatArrayView(Napi::Env env, const float* data, size_t elements) {
    return Napi::Buffer<float>::Copy(env, data, elements * sizeof(float));
}

inline Napi::Buffer<double> DoubleArrayView(Napi::Env env, const double* data, size_t elements) {
    return Napi::Buffer<double>::Copy(env, data, elements * sizeof(double));
}

inline Napi::Object ColorSpaceDataWrapper(Napi::Env env, const photochemistry::ColorSpaceData& c) {
    Napi::Object colorSpace = Napi::Object::New(env);
    colorSpace.Set("illuminant", DoubleArrayView(env, &c.illuminant[0], 2));
    colorSpace.Set("cm", DoubleArrayView(env, &c.cm[0][0], 12));
    colorSpace.Set("cc", DoubleArrayView(env, &c.cc[0][0], 12));
    colorSpace.Set("ab", DoubleArrayView(env, &c.ab[0][0], 12));
    colorSpace.Set("fm", DoubleArrayView(env, &c.fm[0][0], 12));
    return colorSpace;
}

inline Napi::Object ImageMetadataWrapper(Napi::Env env, const photochemistry::ImageMetadata& metadata) {
    Napi::Object imageMetadata = Napi::Object::New(env);
    imageMetadata.Set("make", metadata.make);
    imageMetadata.Set("model", metadata.model);
    imageMetadata.Set("displayMake", metadata.displayMake);
    imageMetadata.Set("displayModel", metadata.displayModel);
    return imageMetadata;
}

inline Napi::Object ImageFormatWrapper(Napi::Env env, const photochemistry::ImageFormat& formatData) {
    Napi::Object imageFormat = Napi::Object::New(env);
    imageFormat.Set("orientation", static_cast<int>(formatData.orientation));
    imageFormat.Set("colorSpace", static_cast<int>(formatData.colorSpace));
    imageFormat.Set("pixelOrder", static_cast<int>(formatData.pixelOrder));
    Napi::Array colorSpaceRecordsArray =
     Napi::Array::New(env, formatData.colorSpaceRecords.size());
    for (size_t i = 0; i < formatData.colorSpaceRecords.size(); i++) {
        colorSpaceRecordsArray[i] = ColorSpaceDataWrapper(
            env, formatData.colorSpaceRecords[i]);
    }
    imageFormat.Set("colorSpaceRecords", colorSpaceRecordsArray);
    imageFormat.Set("metadata", ImageMetadataWrapper(env, formatData.metadata));
    return imageFormat;
}

inline Napi::Object MatWrapper(Napi::Env env, const cv::Mat &mat) {
    Napi::Object wrappedMat = Napi::Object::New(env);
    wrappedMat.Set("rows", mat.rows);
    wrappedMat.Set("cols", mat.cols);
    wrappedMat.Set("type", mat.type());
    wrappedMat.Set("channels", mat.channels());
    wrappedMat.Set("depth", mat.depth());
    if (mat.empty()) {
        wrappedMat.Set("data", Napi::Buffer<uint8_t>::New(env, 0));
    } else if (mat.type() == CV_16U || mat.type() == CV_16F) {
        wrappedMat.Set("data", U16ArrayView(env, mat.ptr<uint16_t>(), mat.total() * mat.channels()));
    } else if (mat.type() == CV_32F) {
        wrappedMat.Set("data", FloatArrayView(env, mat.ptr<float>(), mat.total() * mat.channels()));
    } else if (mat.type() == CV_64F) {
        wrappedMat.Set("data", DoubleArrayView(env, mat.ptr<double>(), mat.total() * mat.channels()));
    } else {
        wrappedMat.Set("data", U8ArrayView(env, mat.ptr<uchar>(), mat.total() * mat.elemSize()));
    }
   
    return wrappedMat;
}

inline Napi::Object ImageWrapper(Napi::Env env, const photochemistry::Image& image) {
   Napi::Object wrappedImage = Napi::Object::New(env);
   wrappedImage.Set("format", ImageFormatWrapper(env, image.format));
   wrappedImage.Set("scale", Napi::Number::New(env, image.scale));
   wrappedImage.Set("mat", MatWrapper(env, image.mat));
   return wrappedImage;
}

using Context = Napi::Reference<Napi::Value>;


inline void callbackWithDouble(Napi::Env env, Napi::Function callback, Context *context, double* value) {
    if (env != nullptr && callback != nullptr) { // Callback function hasn't been deleted
        callback.Call(context->Value(), {Napi::Number::New(env, *value)});
    }
    delete value;
}

inline void callbackWithBytes(Napi::Env env, Napi::Function callback, Context *context, std::vector<uint8_t>* value) {
    if (env != nullptr && callback != nullptr) { // Callback function hasn't been deleted
        callback.Call(context->Value(), {U8ArrayView(env, value->data(), value->size())});
    }
    delete value;
}

inline void callbackWithImage(Napi::Env env, Napi::Function callback, Context *context, photochemistry::Image* image) {
    printf("Running native function for thread safe function\n");
    if (env != nullptr && callback != nullptr) { // Callback function hasn't been deleted
        printf("Calling callback\n");
        if (image != nullptr) {
            callback.Call(context->Value(), {ImageWrapper(env, *image)});
        } else {
            callback.Call(context->Value(), {env.Null()});
        }
    }
    printf("Cleaning up after callback\n");
    delete image;
}

inline void defaultFinalizer(Napi::Env, void*, Context *ctx) {
    printf("Finalizing typed thread safe function\n");
    delete ctx;
}

#endif